/******************************************************************************
humanplayer.cc - class HumanPlayer source file.

Copyright (C) 1999 David Vrabel

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
******************************************************************************/
#include <cctype>
#include <algorithm>

#include "humanplayer.h"

using namespace std;

PlayerAction* HumanPlayer::get_move(const Board& board)
/******************************************************************************
Reads players action interactively.
*/
{
    for(;;) {
        cout << board.get_move_number() << " " << flush;
	
        // Find first token
        Move move;
        switch( get_command( move ) ) {
        case MOVE : 
            if( board.is_valid_move( piece, move ) ) {
                return new MoveAction( move );
            }
            cout << "Illegal move." << endl;
            break;
        case EXIT :
            return new ExitAction();
        case HINT :
            return new HintAction();
            break;
        case REDRAW :
            cout << endl << board << endl;
            break;
        case HELP :
            display_help();
            break;
        default :
            cout << "Invalid move or command. Try `help'." << endl;
            break;
        }
    }
}

inline char my_tolower( char c )
{
    return tolower( c );
}

HumanPlayer::CommandType HumanPlayer::get_command( Move& move )
{
    const string command[4]={
        "exit",
        "hint",
        "redraw",
        "help",
    };

    string input;
    do {
	cin >> input;
    } while( input.length()==0 );

    transform( input.begin(), input.end(), input.begin(), my_tolower );

    // Check for a move
    if( input=="pass" || input=="p" ) {
        move.set_pass();
        return MOVE;
    }
    if( input.length()==2 && isalpha( input[0] ) && isdigit( input[1] ) ) {
        move.set( input[0]-'a', input[1]-'0' );
        return MOVE;
    }

    // Check for command
    for( unsigned i=EXIT; i<=HELP; i++ ) {
        if( input==command[i-EXIT] ) {
            return CommandType( i );
        }
    }
    return ERROR;
}

void HumanPlayer::display_help(void) const
/******************************************************************************
Describes all the commands available on the command line.
*/
{
    cout << "\nCommands available:\n"
	    "  a1          Make move `a1'\n"
	    "  pass or p   Pass this move\n\n"
	    "  exit        Exit reversi\n"
	    "  hint        Obtain hint for next move (if available)\n"
	    "  redraw      Redraw the board\n\n";
}

bool ExitAction::execute(Board& board, Player& player, Player& opponent)
/******************************************************************************
Exits the program.
*/
{
    exit(0);
    return false;
}

bool HintAction::execute(Board& board, Player& player, Player& opponent)
/******************************************************************************
Displays a hint from the opponent.
*/
{
    cout << "Hint: " << opponent.get_hint() << endl;
    return false;
}

