/* computerplayer.h - class ComputerPlayer header file. -*-c++-*-

Copyright (C) 1999 David Vrabel

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
#ifndef COMPUTERPLAYER_H
#define COMPUTERPLAYER_H

#include <string>

#include "player.h"
#include "movetree.h"

// Functions to score a board position.
typedef int ScoringFunc(const Board&, Square::Piece);

extern int SimpleScore(const Board& board, Square::Piece p);
extern int WeightedScore(const Board& board, Square::Piece p);
extern int CornerPhillicScore(const Board& board, Square::Piece piece);

/*
------------------------| class ComputerPlayer |------------------------------
Computer player utilizing recursion to look ahead.
The search depth and the function used to evaluate the score after a move
can be set.

The algorithm used is minimax with alpha-beta cutoff.

The Alpha-Beta Cutoff
---------------------

ALPHA-BETA cutoff is a method for reducing the number of nodes explored in the
Minimax strategy. For the nodes it explores it computes, in addition to the
score, an alpha value and a beta value.

ALPHA value of a node 
    Initially it is the score of that node, if the node is a leaf, otherwise
    it is -infinity. Then at a MAX node it is set to the largest of the scores
    of its successors explored up to now, and at a MIN node to the alpha value
    of its predecessor.

BETA value of a node 
    Initially it is the score of that node, if the node is a leaf, otherwise
    it is +infinity. Then at a MIN node it is set to the smallest of the
    scores of its successors explored up to now, and at a MAX node to the beta
    value of its predecessor. 

It is guaranteed that:

  - The score of a node will always be no less than the alpha value and no
    greater than the beta value of that node. 
  - As the algorithm evolves, the alpha and beta values of a node may change,
    but the alpha value will never decrease, and the beta value will never
    increase. 
  - The alpha value of a node is never less than the alpha value of its
    predecessor. 
  - The beta value of a node is never greater than the beta value of its
    predecessor. 
  - When a node is visited last, its score is set to the alpha value of that 
    node, if it is a MAX node, otherwise it is set to the beta value. 

    function MINIMAX-AB(N, A, B) is
    begin
	Set Alpha value of N to A and Beta value of N to B;
	if N is a leaf then
	    return the estimated score of this leaf
	else
            if N is a Min node then
                For each successor Ni of N loop
		    Let Val be MINIMAX-AB(Ni, Alpha of N, Beta of N);
		    Set Beta value of N to Min{Beta value of N, Val};
		    When Beta value of N <= Alpha value of N then exit loop;
	   	    Return Beta value of N;
	    else
		For each successor Ni of N loop
		    Let Val be MINIMAX-AB(Ni, Alpha of N, Beta of N);
		    Set Alpha value of N to Max{Alpha value of N, Val};
		    When Alpha value of N >= Beta value of N then exit loop;
		Return Alpha value of N;
    end MINIMAX-AB;

The MINIMAX-AB function is called with as parameters the root of the game tree,
-infinity as alpha value, and +infinity as beta value.
*/
static const int MaxSearchDepth=62;
    // Note: Game lasts for a maximum of 62 moves.

class ComputerPlayer : public Player {
private :
    int search_for_next_move(const Board& board, Square::Piece cur_piece,
                             MoveTree::Iterator cur_node, int depth,
                             int alpha, int beta);
    int try_next_move(       Board board, Square::Piece cur_piece,
                             const Move& move, MoveTree::Iterator& cur_node,
                             int depth, int alpha, int beta);
    int try_pass(            Board board, Square::Piece cur_piece,
                             MoveTree::Iterator& cur_node, int depth,
                             int alpha, int beta);
protected :
    ScoringFunc* evaluate_score;
    int          search_depth;
    MoveTree     move_tree;

    virtual bool deep_enough( int depth ) const;
public :
    ComputerPlayer(Square::Piece p) : Player(p,false),
        evaluate_score(SimpleScore), search_depth(1), move_tree() {}
    virtual ~ComputerPlayer() {}

    virtual void          configure(void);
    virtual PlayerAction* get_move(const Board& board);
    virtual std::string   get_hint(void) const;
};

#endif // #ifndef COMPUTERPLAYER_H
