/*****************************************************************************
computerplayer.cc - class ComputerPlayer source code.

Copyright (C) 1999 David Vrabel

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*****************************************************************************/
#include <sstream>
#include <cctype>

#include "computerplayer.h"

using namespace std;

void ComputerPlayer::configure(void)
/*****************************************************************************
Allow user to specify the name, search depth, and scoring method.
*/
{
    Player::configure();

    for(;;) {
        cout << "Enter search depth: " << flush;
        string temp;
        getline(cin, temp);
        search_depth=atoi(temp.c_str());

        if( search_depth>0 && search_depth<=MaxSearchDepth ) {
            break;
        }
        cout << "Invalid value! Valid range: 1..62" << endl;
    }

    cout << "Select scoring method ([S]imple, [W]eighted, [C]orner-phillic): "
         << flush;
    evaluate_score=NULL;
    while( evaluate_score==NULL ) {
        string temp;
        getline(cin,temp);
        switch( tolower(temp[0]) ) {
        case 's' :
            evaluate_score=SimpleScore;
            break;
        case 'w' :
            evaluate_score=WeightedScore;
            break;
        case 'c' :
            evaluate_score=CornerPhillicScore;
            break;
        }
    }
}

PlayerAction* ComputerPlayer::get_move(const Board& board)
/****************************************************************************
Finds the best valid move.
*/
{
    move_tree.destroy();
    search_for_next_move(board, piece, move_tree.get_root(), 0,
                         numeric_limits<int>::min(), 
                         numeric_limits<int>::max());

    MoveTree::Iterator best=move_tree.get_root();
    best.move_up_best_branch();

    return new MoveAction(*best);
}

int ComputerPlayer::search_for_next_move(const Board& board,
                                         Square::Piece cur_piece, MoveTree::Iterator cur_node,
                                         int depth, int alpha, int beta)
/*****************************************************************************
Selects the best move from all possible moves.
Returns the best score so far.
*/
{
    if( deep_enough( depth ) ) {
        return evaluate_score(board, piece);
    }
    // Try a pass first
    int score=try_pass(board, cur_piece, cur_node, depth+1, alpha, beta);
    move_tree.reset_best_branch(cur_node);

    Move move(0, 0);
    if( piece==cur_piece ) { // MAX player
        if( score>alpha ) {
            alpha=score;
        }
        for( move.x=0; move.x<Board::x_size; move.x++ ) {
            for( move.y=0; move.y<Board::y_size; move.y++ ) {
                if( alpha>=beta ) {
                    return alpha;
                }
                score=try_next_move(board, cur_piece, move, cur_node, depth+1, alpha, beta);
                if( score>alpha ) {
                    alpha=score;
                    move_tree.reset_best_branch(cur_node);
                }
            }
        }
        return alpha;
    } else { // MIN player
        if( score<beta ) {
            beta=score;
        }
        for( move.x=0; move.x<Board::x_size; move.x++ ) {
            for( move.y=0; move.y<Board::y_size; move.y++ ) {
                if( beta<=alpha ) {
                    return beta;
                }
                score=try_next_move(board, cur_piece, move, cur_node, depth+1, alpha, beta);
                if( score<beta ) {
                    beta=score;
                    move_tree.reset_best_branch(cur_node);
                }
            }
        }
        return beta;
    }
}

int ComputerPlayer::try_next_move(Board board, Square::Piece cur_piece,
                                  const Move& move, MoveTree::Iterator& cur_node, int depth,
                                  int alpha, int beta )
/*****************************************************************************
Trys `move' returning the best score for that move.
*/
{
    if( board.get_square(move.x, move.y).get_piece()!=Square::blank ) {
        return cur_piece==piece?
            numeric_limits<int>::min():
            numeric_limits<int>::max();
    }
    unsigned initial_num_pieces=board.get_num_pieces(cur_piece);
    board.do_move(cur_piece, move);

    // If no swops occurred...invalid move
    if( initial_num_pieces+1==board.get_num_pieces(cur_piece) ) {
        return cur_piece==piece?
            numeric_limits<int>::min():
            numeric_limits<int>::max();
    }
    MoveTree::Iterator next_node=move_tree.add_branch(move, cur_node);

    return search_for_next_move(board, OtherPiece(cur_piece),
                                next_node, depth, alpha, beta);
}

int ComputerPlayer::try_pass(Board board, Square::Piece cur_piece,
                             MoveTree::Iterator& cur_node, int depth,
                             int alpha, int beta)
/*****************************************************************************
Trys a pass returning the best score for that move.
*/
{
    Move move;
    move.set_pass();
    board.do_move(cur_piece, move);
    MoveTree::Iterator next_node=move_tree.add_branch(move, cur_node);
    if( board.is_game_over() ) {
        return 1000*(board.get_num_pieces(piece)
                     -board.get_num_pieces(OtherPiece(piece)));
    }

    return search_for_next_move(board, OtherPiece(cur_piece),
                                next_node, depth, alpha, beta);
}

bool ComputerPlayer::deep_enough( int depth ) const
{
    return depth==search_depth;
}


string ComputerPlayer::get_hint(void) const
/*****************************************************************************
Returns what the computer thought was the next best move in the format:
"Pass", "a1", or "Not available" if no hint is available.
*/
{
    MoveTreeIterator hint=move_tree.get_root();
    for( unsigned i=0; i<2; i++ ) {
        hint.move_up_best_branch();
    }

    if( hint ) {
        stringstream s;
        s << *hint;
        return s.str();
    }
    return "Not available";
}


/*
---------------------| Functions to score the board |-------------------------
*/

extern int SimpleScore(const Board& board, Square::Piece p)
/*****************************************************************************
Calculates the difference between the number of 'p' pieces and the other
pieces.
*/
{
    return board.get_num_pieces(p)-board.get_num_pieces(OtherPiece(p));
}

extern int WeightedScore(const Board& board, Square::Piece p)
/*****************************************************************************
Calculates the difference between the number of 'p' pieces and the other
pieces. Taking into account weighting of various squares.
*/
{
    static int weightings[Board::x_size][Board::y_size]={
        {100,50,50,50,50,50,50,100},
        { 50, 1, 1, 1, 1, 1, 1, 50},
        { 50, 1, 1, 1, 1, 1, 1, 50},
        { 50, 1, 1, 1, 1, 1, 1, 50},
        { 50, 1, 1, 1, 1, 1, 1, 50},
        { 50, 1, 1, 1, 1, 1, 1, 50},
        { 50, 1, 1, 1, 1, 1, 1, 50},
        {100,50,50,50,50,50,50,100},
    };
    int score=0;
    for( unsigned x=0; x<Board::x_size; x++ ) {
        for( unsigned y=0; y<Board::y_size; y++ ) {
            if( board.get_square(x, y).get_piece()==Square::blank ) {
                continue;
            }
            if( board.get_square(x, y).get_piece()==p ) {
                score+=weightings[x][y];
            } else {
                score-=weightings[x][y];
            }
        }
    }
    return score;
}

struct Coord {
    int x;
    int y;
};
struct CornerAdj {
    Coord corner;
    Coord adjacent[3];
};

extern int CornerPhillicScore(const Board& board, Square::Piece piece)
/******************************************************************************
Calculates the score of the board for a player taking into account the
desirablity of the corner squares, and the undesirability of the squares next 
to the corners (if they aren't occupied).
*/
{
    static CornerAdj Corners[4]={
        {{0,0},{{0,1},{1,1},{1,0}}},
        {{0,7},{{1,7},{1,6},{0,6}}},
        {{7,7},{{7,6},{6,6},{6,7}}},
        {{7,0},{{6,0},{6,1},{7,1}}}
    };      
    int score=board.get_num_pieces(piece)-board.get_num_pieces(OtherPiece(piece)); 

    for( CornerAdj* c=Corners; c<Corners+4; c++ ) {
        Square::Piece p=board.get_square(c->corner.x, c->corner.y).get_piece();
        if( p==piece ) {
            score+=100;
        } else if( p==OtherPiece(piece) ) {
            score-=100;
        } else {
            for( Coord* a=c->adjacent; a<c->adjacent+3; a++ ) {
                p=board.get_square(a->x, a->y).get_piece();
                if( p==piece ) {
                    score-=50;
                } else if( p==OtherPiece(piece) ) {
                    score+=50;
                }
            }
        }
    }
    return score;
}

