/* computer2.cc - class ComputerPlayer source code.

Copyright 2002 David Vrabel

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
#include <cctype>
#include <ctime>
#include <sys/time.h>
#include <iomanip>

#include "computer2.h"

using namespace std;

void ComputerPlayer2::configure(void)
/*****************************************************************************
Allow user to specify the name, search depth, and scoring method.
*/
{
    Player::configure();

    for(;;) {
        cout << "Enter maximum search time per move (seconds): ";
        string temp;
        getline( cin, temp );
        search_time=atof( temp.c_str() );
        if( search_time>0.0 ) {
            break;
        }
        cout << "Invalid value. Must be greater than 0.\n";
    }

    evaluate_score=NULL;
    do {
        cout << "Select scoring method ([S]imple, [W]eighted, [C]orner-phillic): "
             << flush;
        string temp;
        getline(cin,temp);
        switch( tolower(temp[0]) ) {
        case 's' :
            evaluate_score=SimpleScore;
            break;
        case 'w' :
            evaluate_score=WeightedScore;
            break;
        case 'c' :
            evaluate_score=CornerPhillicScore;
            break;
        default :
            cout << "Invalid entry.\n";
            break;
        }
    } while( evaluate_score==NULL );
}

PlayerAction* ComputerPlayer2::get_move(const Board& board)
/****************************************************************************
Finds the best valid move.

clock() returns the processor time used and not the system time elapsed so it
may not be appropriate here.
*/
{
    struct timeval tv;
    double end_time;
    PlayerAction* action;

    gettimeofday( &tv, 0 );
    start_time = double(tv.tv_sec) + double(tv.tv_usec)/1.0e6;

    /* Search to a maximum depth of 2*number of blanks left+2 to take account
       of move sequences like pmpmpmpp etc. (p=pass m=move). */
    for( search_depth=1;
         search_depth<=2*board.get_num_pieces( Square::blank )+2;
         search_depth++ ) {
        try {
            action=ComputerPlayer::get_move( board );
            hint=ComputerPlayer::get_hint();
        }
        catch( OutOfTime ) {
            search_depth--;
            break;
        }
        gettimeofday( &tv, 0 );
        end_time = double(tv.tv_sec) + double(tv.tv_usec)/1.0e6;
    }
    cout << '\n' << name << " looked " << search_depth << " moves ahead in "
         << fixed << setprecision(2) << end_time-start_time << " seconds.\n\n";

    return action;
}


bool ComputerPlayer2::deep_enough( int depth ) const
{
    struct timeval tv;
    double cur_time;

    gettimeofday( &tv, 0 );
    cur_time = double(tv.tv_sec) + double(tv.tv_usec)/1.0e6;

    if( depth > 1 && cur_time-start_time > search_time ) {
        throw OutOfTime();
    }
    return ComputerPlayer::deep_enough( depth );
}


string ComputerPlayer2::get_hint() const
{
    return hint;
}
