/*board.h - class Board and related classes (Move and Square). -*-c++-*-

Copyright (C) 1999 David Vrabel

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
#ifndef BOARD_H
#define BOARD_H

#include <string>
#include <iostream>

/*
----------------------------| class Move |------------------------------------
Used to comunicate the move selected by a player to the board for processing.
*/

// Forward declare friends for Move
class Board;
class Player;
class HumanPlayer;
class ComputerPlayer;
class AlphaBetaPlayer;

class Move {
protected :
    bool pass;
    unsigned x;
    unsigned y;
public :
    Move(void) : pass(false), x(0), y(0) {}
    Move(int _x, int _y) : pass(false), x(_x), y(_y) {}

    bool is_pass(void) const { return pass; }
    int get_x(void) const { return x; }
    int get_y(void) const { return y; }

    void set_pass(void) { pass=true; }
    void set(int _x, int _y) { pass=false; x=_x; y=_y; }

    friend std::ostream& operator<<(std::ostream& s, const Move& move);

    friend class Board;
    friend class Player;
    friend class HumanPlayer;
    friend class ComputerPlayer;
    friend class AlphaBetaPlayer;
};

/*
-----------------------------| class Square |--------------------------------
One square of the reversi board.
Defines enum Piece which is used to identify players' pieces.
*/

class Square {
public :
    enum Piece {
        blank, x, o
    };
protected :
    Piece piece;
public :
    Square(void) : piece(blank) {}
    Square(Piece p) : piece(p) {}

    Piece get_piece(void) const { return piece; }
    void set_piece(Piece p) { piece=p; }
};

inline Square::Piece OtherPiece(Square::Piece p)
{
    if( p==Square::blank ) {
        return Square::blank;
    }
    return p==Square::x?Square::o:Square::x;
}

/*
----------------------------| class Board |-----------------------------------
The reversi board.
*/

class Board {
public :
    static const unsigned x_size=8; // Size of board
    static const unsigned y_size=8;
private :
    int num_sandwiched(Square::Piece piece, unsigned x, unsigned y,
                       unsigned i, unsigned j) const;
protected :
    Square   board[x_size][y_size];
    unsigned num_pieces[3]; // Number of each type of piece
    unsigned num_passes;   // Number of passes in a row
    unsigned num_moves;   // Number of single turn moves carried out
    char     piece_symbol[3];
public :
    Board(void);
    ~Board() {}

    const Square& get_square(unsigned x, unsigned y) const { return board[x][y]; }
    unsigned get_num_pieces(Square::Piece piece) const { return num_pieces[piece]; }
    unsigned get_num_moves(void) const { return num_moves; }

    bool is_valid_move(Square::Piece piece, const Move& move) const;
    void do_move(Square::Piece piece, const Move& move);

    bool is_game_over(void) const { return num_passes>1; }
    Square::Piece get_winner(void) const;

    std::string get_move_number(void) const;
    friend std::ostream& operator<< (std::ostream& o, const Board& board);
};

#endif // #ifdef BOARD_H

