/*****************************************************************************
board.cc - class Board and related classes (Move and Square) source code.

Copyright (C) 1999 David Vrabel

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*****************************************************************************/
#include <string>
#include <iostream>
#include <sstream>

#include"board.h"

using namespace::std;

ostream& operator<< (ostream& o, const Move& move)
/*****************************************************************************
Output `move' in the format a1
*/
{
    if( move.pass ) {
        o << "Pass";
    } else {
        o << char(move.x+'a') << move.y;
    }
    return o;
}


Board::Board()
/*****************************************************************************
Initialize the board to blank. Set the centre squares.
*/
: num_passes(0), num_moves(0)
{
    for( unsigned x=0; x<x_size; x++ ) {
        for( unsigned y=0; y<y_size; y++ ) {
            board[x][y].set_piece(Square::blank);
        }
    }

    // Set initial pattern in centre.
    board[3][3].set_piece(Square::x);
    board[4][4].set_piece(Square::x);
    board[3][4].set_piece(Square::o);
    board[4][3].set_piece(Square::o);
    
    num_pieces[Square::blank]=x_size*y_size-4;
    num_pieces[Square::x]=2;
    num_pieces[Square::o]=2;

    piece_symbol[Square::blank]='.';
    piece_symbol[Square::x]='X';
    piece_symbol[Square::o]='O';
}


// Vector indicating the eight directions from a square
static struct DirectionVector {
    int x;
    int y;
} Directions[8]={
    {-1,-1},{0,-1},{1,-1},{1,0},{1,1},{0,1},{-1,1},{-1,0}
};


/*****************************************************************************
Check if a `move' is a valid move.
*/
bool Board::is_valid_move( Square::Piece piece, const Move& move ) const
{
    if( move.pass ) {
        return true;
    }

    if( move.x<0 || move.x>=x_size || move.y<0 || move.y>=y_size ) {
        return false;
    }

    if( board[move.x][move.y].get_piece()!=Square::blank ) {
        return false;
    }

    // Check that move will lead to a sandwich.
    for( DirectionVector* d=Directions; d<Directions+8; d++ ) {
        if( num_sandwiched(piece, move.x, move.y, d->x, d->y)>0 ) {
            return true;
        }
    }
    return false;
}

/*****************************************************************************
Counts the number of sandwiched pieces between (x,y) in direction (i,j).
*/
int Board::num_sandwiched(Square::Piece piece, unsigned x, unsigned y,
                          unsigned i, unsigned j) const
{
    int num_sandwiched=0;

    for( x+=i,y+=j;;x+=i,y+=j ) {
        if( x<0 || x>=x_size || y<0 || y>=y_size ) {
            return 0;
        }
        if( board[x][y].get_piece()==Square::blank ) {
            return 0;
        }
        if( board[x][y].get_piece()==piece ) {
            return num_sandwiched;
        }
        num_sandwiched++;
    }
}

void Board::do_move(Square::Piece piece, const Move& move)
/*****************************************************************************
Does a move, swopping all pieces that are sandwiched.
*/
{
    num_moves++;
    if( move.pass ) {
        num_passes++;
        return;
    }
    for( int n=0; n<8; n++ ) {
        int i=Directions[n].x;
        int j=Directions[n].y;
        int sandwiched=num_sandwiched(piece,move.x,move.y,i,j);
        for( int swopped=1; swopped<=sandwiched; swopped++ ) {
            board[move.x+swopped*i][move.y+swopped*j].set_piece(piece);
        }
        num_pieces[piece]+=sandwiched;
        num_pieces[OtherPiece(piece)]-=sandwiched;
    }
    board[move.x][move.y].set_piece(piece);
    num_pieces[piece]++;
    num_pieces[Square::blank]--;
    num_passes=0;
}

Square::Piece Board::get_winner( void ) const
/*****************************************************************************
Returns the winning piece (ie one with most number of pieces) or Square::blank
for a draw.
*/
{
    if( num_pieces[Square::x]>num_pieces[Square::o] ) {
        return Square::x;
    }
    if( num_pieces[Square::o]>num_pieces[Square::x] ) {
        return Square::o;
    }
    return Square::blank;
}


string Board::get_move_number(void) const
/*****************************************************************************
Returns a string containing the move number eg: "1:" or "1: ..."
*/
{
    stringstream str;
    str << num_moves/2+1 << (num_moves&1?": ...":":");
    return str.str();
}

ostream& operator<< (ostream& o, const Board& board)
/*****************************************************************************
Returns a string containing the move number eg: "1:" or "1: ..."
*/
{
    o << board.piece_symbol[Square::x] << ": " << board.num_pieces[Square::x]
      << "  " << board.piece_symbol[Square::o] << ": "
      << board.num_pieces[Square::o] << endl << endl;

    for( int y=Board::y_size-1; y>=0; y-- ) {
        o << y << " ";
        for( unsigned x=0; x<Board::x_size; x++ ) {
            o << board.piece_symbol[board.board[x][y].get_piece()] << " ";
        }
        o << endl;
    }
    o << "  a b c d e f g h" << endl;
    return o;
}
