# FlexiRPG -- Images identified by UUID
#
# Copyright (C) 2017 David Vrabel
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
import base64
import os.path
import StringIO
import wx

import orpg.dirpath

class Image(object):
    def __init__(self, uuid, size = None):
        """Create an image with a UUID and (optionally) size.

        """
        self.uuid = uuid
        self.size = size
        self.data = None
        self.data64 = None
        self._hooks = []

        self.images_dir = os.path.join(orpg.dirpath.dir_struct["user"], "images")

    def set_image(self, data):
        assert not self.has_image()

        self.data = data

        # Save the image to the file system cache.
        try:
            if not os.path.exists(self.images_dir):
                os.mkdir(self.images_dir)
            cache_file = os.path.join(self.images_dir, "%s.png" % self.uuid)
            if not os.path.exists(cache_file):
                f = open(cache_file, "wb")
                f.write(data)
                f.close()
        except (OSError, IOError):
            # Silently ignore errors when updating the cache file
            pass

        for hook in self._hooks:
            hook(self)

    def has_image(self):
        return self.data != None

    def __get_width(self):
        return self.size[0]

    def __get_height(self):
        return self.size[1]

    width = property(__get_width)
    height = property(__get_height)

    def add_hook(self, hook):
        self._hooks.append(hook)

    def del_hook(self, hook):
        self._hooks.remove(hook)

    def base64(self):
        if not self.data64:
            self.data64 = base64.b64encode(self.data)
        return self.data64

    def set_image_from_base64(self, b64):
        self.set_image(base64.b64decode(b64))

class DisplayImage(Image):
    """An image that can be displayed in a wx.Widget -- it has a wx.Image.
    """
    def __init__(self, uuid, size = None):
        Image.__init__(self, uuid, size)

        if size:
            # Create a place holder image to be displayed before the real
            # image has been fetched.
            pixels = [(0x0, 0x0, 0x0, 0x0)] * self.width * self.height

            for x in range(self.width):
                for y in range(self.height):
                    if x > 2 and x < self.width-3 and y > 2 and y < self.height-3:
                        continue
                    if (x + y) % 6 < 3:
                        colour = (230, 242, 64, 255) # Yellowish
                    else:
                        colour = (0, 0, 0, 255) # Black
                    pixels[x + y*self.height] = colour

            s = ""
            for p in pixels:
                s += "%c%c%c%c" % (p[0], p[1], p[2], p[3])

            self.bitmap = wx.Bitmap.FromBufferRGBA(self.width, self.height, s)
            self.wximage = self.bitmap.ConvertToImage()

    def set_image(self, data):
        self.wximage = wx.Image(StringIO.StringIO(data), "image/png")
        self.size = (self.wximage.GetWidth(), self.wximage.GetHeight())
        self.bitmap = wx.Bitmap(self.wximage)

        Image.set_image(self, data)


class ServerImage(Image):
    """An image in the server -- it keeps track of who requested it so it
    can send IMAGEDATA messages to clients.

    """

    def __init__(self, uuid, size=None):
        Image.__init__(self, uuid, size)

        self.provider = None
        self.requesters = []

    def add_requester(self, provider, player_id):
        self.provider = provider
        self.requesters.append(player_id)

    def has_requesters(self):
        return self.requesters != []

    def set_image(self, data):
        Image.set_image(self, data)

        for player_id in self.requesters:
            self.provider.send_imagedata(player_id, self)
        self.requesters = []
