# Tree icon management.
#
# Copyright (C) 2011 David Vrabel
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

from orpg.orpg_wx import *
from orpg.orpg_windows import *
from orpg.orpgCore import open_rpg
import orpg.dirpath
import math

class node_icons(wx.ImageList):
    def __init__(self):
        wx.ImageList.__init__(self, 16, 16, False)

        self._icons = {}
        self._images = []
        self._names = []

        try:
            xml_dom = orpg.orpg_xml.parse_file(orpg.dirpath.dir_struct["icon"] + "icons.xml")
        except IOError:
            return
        except:
            raise

        for n in xml_dom.getElementsByTagName('icon'):
            key = n.getAttribute("name")
            path = orpg.dirpath.dir_struct["icon"] + n.getAttribute("file")
            try:
                img = wx.Image(path, wx.BITMAP_TYPE_ANY)
                self._images.append(img)
                self._icons[key] = self.Add(wx.BitmapFromImage(img))
                self._names.append(key)
            except IOError:
                pass

    def __getitem__(self, icon_name):
        return self._icons.get(icon_name, 0)

    def count(self):
        return len(self._images)

    def name(self, index):
        return self._names[index]

    def image(self, index):
        return self._images[index]

    def largest_size(self):
        (w, h) = (0, 0)
        for i in range(self.count()):
            (nw, nh) = self._images[i].GetSize()
            if nw > w:
                w = nw
            if nh > h:
                h = nh
        return (w, h)

IconSelectedType = wx.NewEventType()
EVT_ICON_SELECTED = wx.PyEventBinder(IconSelectedType, 1)

class IconSelectedEvent(wx.PyCommandEvent):
    def __init__(self, evt_type, id_):
        wx.PyCommandEvent.__init__(self, evt_type, id_)

class icon_picker(wx.BitmapButton):
    def __init__(self, parent, icons, selected = 0):
        wx.BitmapButton.__init__(self, parent, wx.ID_NONE,
                                wx.BitmapFromImage(icons.image(selected)))

        self._icons = icons
        self._popup = icon_popup(self, icons)

        self.Bind(wx.EVT_BUTTON, self.on_button)
        self._popup.Bind(EVT_ICON_SELECTED, self.on_icon_selected)

    def on_button(self, evt):
        (x, y) = self.GetScreenPosition()
        (w, h) = self.GetSize()
        self._popup.Position((x, y), (0, h))
        self._popup.Popup()

    def on_icon_selected(self, evt):
        self.SetBitmapLabel(wx.BitmapFromImage(self._icons.image(evt.GetId())))
        evt.Skip()

class icon_popup(wx.PopupTransientWindow):

    COLUMNS_MAX_DEFAULT = 15
    WINDOW_BORDER = 1
    ICON_BORDER = 3

    def __init__(self, parent, icons):
        wx.PopupTransientWindow.__init__(self, parent, style = wx.BORDER_SIMPLE)

        self._icons = icons
        self._icon_size = max(icons.largest_size())
        self._max_columns = self.COLUMNS_MAX_DEFAULT
        self._selected = None

        self.SetBackgroundStyle(wx.BG_STYLE_CUSTOM)

        (self.cols, self.rows) = self._grid_size()

        self._spacing = self._icon_size + 2 * self.ICON_BORDER + self.WINDOW_BORDER

        w = self.cols * self._spacing + self.WINDOW_BORDER
        h = self.rows * self._spacing + self.WINDOW_BORDER

        self.SetSize((w, h))

        self.Bind(wx.EVT_PAINT, self.on_paint)
        self.Bind(wx.EVT_MOUSE_EVENTS, self.on_mouse)
        self.Bind(wx.EVT_KEY_DOWN, self.on_key_down)

    def _grid_size(self):
        num_icons = self._icons.count()

        # Aim for a roughly square grid of icons, but not too wide.
        cols = int(math.sqrt(num_icons) + 0.5)
        if cols > self._max_columns:
            cols = self._max_columns
        rows = (num_icons + cols - 1) / cols

        return (cols, rows)

    def on_paint(self, evt):
        (w, h) = self.GetClientSize()
        sp = self._spacing
        ib = self.ICON_BORDER
        wb = self.WINDOW_BORDER

        dc = wx.AutoBufferedPaintDC(self)
        rn = wx.RendererNative.GetDefault()

        dc.SetBrush(wx.Brush(wx.WHITE))
        dc.SetPen(wx.Pen(wx.LIGHT_GREY))
        dc.DrawRectangle(0, 0, w, h)

        for i in range(self._icons.count()):
            x = wb + i % self.cols * sp
            y = wb + i / self.cols * sp
            if i == self._selected:
                rn.DrawItemSelectionRect(self, dc, (x, y, sp - wb, sp - wb),
                                         wx.CONTROL_SELECTED | wx.CONTROL_FOCUSED)
            if i < self._icons.count():
                bmp = wx.BitmapFromImage(self._icons.image(i))
                dc.DrawBitmap(bmp, x + ib, y + ib, True)

        dc.SetPen(wx.Pen(wx.LIGHT_GREY))
        for c in range(1,self.cols):
            x = wb + c * sp
            dc.DrawLine(x-1, 1, x-1, h - 1)
            for r in range(1,self.rows):
                y = wb + r * sp
                dc.DrawLine(1, y-1, w-1, y-1)

    def on_key_down(self, evt):
        if evt.GetKeyCode() == wx.WXK_ESCAPE:
            self.Dismiss()

    def on_mouse(self, evt):
        sel = None

        (w, h) = self.GetClientSize()
        (x, y) = evt.GetPosition()
        wb = self.WINDOW_BORDER
        x -= wb
        y -= wb
        if 0 <= x < (w - wb) and 0 <= y < (h - wb):
            c = x / self._spacing
            r = y / self._spacing
            i = c + r * self.cols
            if i < self._icons.count():
                sel = i
        if sel != self._selected:
            self._selected = sel
            self.Refresh()

        if  evt.LeftDown() and self._selected != None:
            evt = IconSelectedEvent(IconSelectedType, i)
            wx.PostEvent(self, evt)
            self.Dismiss()

if __name__ == "__main__":
    import xml.dom.minidom

    class icon_picker_test(wx.App):
        def OnInit(self):
            frame = wx.Frame(None, -1, "Icon Combo Test")

            orpg.dirpath.dir_struct["icon"] = "images/"

            self.icons = node_icons()

            btn = icon_picker(frame, self.icons);

            hbox = wx.BoxSizer(wx.HORIZONTAL)
            hbox.Add(btn, 1, wx.EXPAND)

            frame.SetSizerAndFit(hbox)

            btn.Bind(EVT_ICON_SELECTED, self.on_selected)

            frame.Show(True)
            self.SetTopWindow(frame)
            return True

        def on_selected(self, evt):
            print "Clicked: %d" % evt.GetId()

    app = icon_picker_test(0)
    app.MainLoop()
