# FlexiRPG -- stack of z-ordered whiteboard objects.
#
# Copyright (C) 2009-2010 David Vrabel
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
from base import *
from whiteboard_object import WhiteboardObject

class WhiteboardStack(object):
    def __init__(self):
        self.stack = {}
        self.top_z = 0

    def append(self, obj):
        self._set(self.top_z, obj)
        self.top_z += 1

    def insert(self, obj, new_z):
        if new_z >= self.top_z:
            self.append(obj)
        else:
            self._shift_up(new_z, self.top_z)
            self._set(new_z, obj)
            self.top_z += 1

    def remove(self, obj):
        self.top_z -= 1
        self._shift_down(obj.z_order, self.top_z)
        del self.stack[self.top_z]

    def move(self, obj, new_z):
        if obj.z_order != new_z:
            self.remove(obj)
            self.insert(obj, new_z)

    def clear(self):
        self.stack.clear()
        self.top_z = 0

    def raise_(self, obj):
        new_z = obj.z_order + 1
        if new_z >= self.top_z:
            return
        self._swap(obj, self.stack[new_z])

    def lower(self, obj):
        new_z = obj.z_order - 1
        if new_z < 0:
            return
        self._swap(obj, self.stack[new_z])

    def raise_to_top(self, obj):
        new_z = self.top_z - 1
        self._shift_down(obj.z_order, new_z)
        self._set(new_z, obj)

    def lower_to_bottom(self, obj):
        new_z = 0
        self._shift_up(new_z, obj.z_order)
        self._set(new_z, obj)

    def _set(self, z, obj):
        self.stack[z] = obj
        obj.z_order = z

    def _shift_up(self, begin, end):
        for i in range(end, begin, -1):
            self._set(i, self.stack[i-1])

    def _shift_down(self, begin, end):
        for i in range(begin, end):
            self._set(i, self.stack[i+1])

    def _swap(self, a, b):
        t = a.z_order
        a.z_order = b.z_order
        b.z_order = t
        self.stack[a.z_order] = a;
        self.stack[b.z_order] = b

    def __len__(self):
        return len(self.stack)

    def __iter__(self):
        for obj in sorted(self.stack.itervalues(), _cmp_z_order):
            yield obj

def _cmp_z_order(first, second):
    f = first.z_order
    s = second.z_order
    if f == s:
        return 0
    elif f < s:
        return -1
    else:
        return 1

if __name__ == "__main__":
    class test_obj(WhiteboardObject):
        def __init__(self, num):
            self.num = num
            WhiteboardObject.__init__(self, "id-" + str(num))
    
    def check(stack, s, result):
        failed = False
        print s + ": ",
        if len(stack) != len(result):
            failed = True
        else:
            for obj, num in zip(stack, result):
                if obj.num != num:
                    failed = True
        if failed:
            print "FAILED"
            s = ""
            for o in stack:
                s += o.id + "(" + str(o.z_order) + "),"
            print s
        else:
            print "Passed"

    obj = {}
    stack = WhiteboardStack()
    for i in range(0, 11):
        obj[i] = test_obj(i)

    for i in range(0, 10):
        stack.append(obj[i])
    check(stack, "append()", [0, 1, 2, 3, 4, 5, 6, 7, 8, 9])

    stack.raise_(obj[9])
    check(stack, "raise_() last", [0, 1, 2, 3, 4, 5, 6, 7, 8, 9])

    stack.lower(obj[0])
    check(stack, "lower() first", [0, 1, 2, 3, 4, 5, 6, 7, 8, 9])

    stack.raise_(obj[2])
    check(stack, "raise_()", [0, 1, 3, 2, 4, 5, 6, 7, 8, 9])

    stack.lower(obj[7])
    check(stack, "lower_()", [0, 1, 3, 2, 4, 5, 7, 6, 8, 9])

    stack.insert(obj[10], 4)
    check(stack, "insert()", [0, 1, 3, 2, 10, 4, 5, 7, 6, 8, 9])

    stack.remove(obj[2])
    check(stack, "remove()", [0, 1, 3, 10, 4, 5, 7, 6, 8, 9])

    stack.raise_to_top(obj[1])
    check(stack, "raise_to_top()", [0, 3, 10, 4, 5, 7, 6, 8, 9, 1])

    stack.lower_to_bottom(obj[9])
    check(stack, "lower_to_bottom()", [9, 0, 3, 10, 4, 5, 7, 6, 8, 1])

    stack.move(obj[0], 7)
    check(stack, "move()", [9, 3, 10, 4, 5, 7, 6, 0, 8, 1])
