from orpg.orpgCore import open_rpg
import wx

class WhiteboardText:
    def __init__(self, id, text_string, pos, style, pointsize, weight, color="#000000"):
        self.text_string = text_string
        self.id = id
        self.weight = int(weight)
        self.pointsize = int(pointsize)
        self.style = int(style)
        self.textcolor = wx.Color()
        self.textcolor.SetFromName(color)
        self.posx = pos.x
        self.posy = pos.y

        self.font = wx.Font(self.pointsize, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL,
                            wx.FONTWEIGHT_NORMAL, False,
                            open_rpg.get_component('settings').get_setting('defaultfont'))

        self.highlighted = False
        self.is_updated = False

    @classmethod
    def create_from_dom(cls, xml_dom):
        text_string = xml_dom.getAttribute('text_string')
        style = xml_dom.getAttribute('style')
        pointsize = xml_dom.getAttribute('pointsize')
        weight = xml_dom.getAttribute('weight')
        color = xml_dom.getAttribute('color')
        id = xml_dom.getAttribute('id')
        posx = xml_dom.getAttribute('posx')
        posy = xml_dom.getAttribute('posy')
        pos = wx.Point(int(posx), int(posy))

        text = WhiteboardText(id, text_string, pos, style, pointsize, weight, color)

        return text

    def highlight(self, highlight=True):
        self.highlighted = highlight

    def move(self, delta):
        self.posx += delta.x
        self.posy += delta.y
        self.is_updated = True

    def set_text_props(self, text_string, style, point, weight, color):
        self.text_string = text_string
        self.textcolor = color

        self.style = int(style)
        self.font.SetStyle(self.style)

        self.pointsize = int(point)
        self.font.SetPointSize(self.pointsize)

        self.weight = int(weight)
        self.font.SetWeight(self.weight)

        self.is_updated = True

    def hit_test(self, pt, dc):
        rect = self.get_rect(dc)
        return rect.InsideXY(pt.x, pt.y)

    def get_rect(self, dc):
        dc.SetFont(self.font)
        (w,h,d,v) = dc.GetFullTextExtent(self.text_string)
        return wx.Rect(self.posx, self.posy, w, h)

    def draw(self, parent, dc, op=wx.COPY):
        # Draw text
        dc.SetTextForeground(self.textcolor)
        dc.SetFont(self.font)
        dc.DrawText(self.text_string, self.posx, self.posy)
        dc.SetTextForeground(wx.Colour(0,0,0))

        # Selection handles
        if self.highlighted:
            dc.SetPen(wx.BLACK_PEN)
            dc.SetBrush(wx.LIGHT_GREY_BRUSH)

            (x,y,w,h) = self.get_rect(dc)
            dc.DrawRectangle(x-7, y-7, 7, 7)
            dc.DrawRectangle(x+w, y-7, 7, 7)
            dc.DrawRectangle(x-7, y+h, 7, 7)
            dc.DrawRectangle(x+w, y+h, 7, 7)

            
    def toxml(self, action="update"):
        if action == "del":
            xml_str = "<text action='del' id='" + self.id + "'/>"
            return xml_str

        xml_str = "<text"

        xml_str += " action='" + action + "'"
        xml_str += " id='" + self.id + "'"

        if self.pointsize != None:
            xml_str += " pointsize='" + str(self.pointsize) + "'"

        if self.style != None:
            xml_str += " style='" + str(self.style) + "'"

        if self.weight != None:
            xml_str += " weight='" + str(self.weight) + "'"

        if self.posx != None:
            xml_str+= " posx='" + str(self.posx) + "'"

        if not (self.posy is None):
            xml_str += " posy='" + str(self.posy) + "'"

        if self.text_string != None:
            xml_str+= " text_string='" + self.text_string + "'"

        if self.textcolor != None:
            xml_str += " color='" + self.textcolor.GetAsString(wx.C2S_HTML_SYNTAX) + "'"

        xml_str += "/>"

        if (action == "update" and self.is_updated) or action == "new":
            self.isUpdated = False
            return xml_str
        else:
            return ''

    def takedom(self, xml_dom):
        self.text_string = xml_dom.getAttribute("text_string")

        if xml_dom.hasAttribute("posy"):
            self.posy = int(xml_dom.getAttribute("posy"))

        if xml_dom.hasAttribute("posx"):
            self.posx = int(xml_dom.getAttribute("posx"))

        if xml_dom.hasAttribute("weight"):
            self.weight = int(xml_dom.getAttribute("weight"))
            self.font.SetWeight(self.weight)

        if xml_dom.hasAttribute("style"):
            self.style = int(xml_dom.getAttribute("style"))
            self.font.SetStyle(self.style)

        if xml_dom.hasAttribute("pointsize"):
            self.pointsize = int(xml_dom.getAttribute("pointsize"))
            self.font.SetPointSize(self.pointsize)

        if xml_dom.hasAttribute("color") and xml_dom.getAttribute("color") != '':
            self.textcolor.SetFromName(xml_dom.getAttribute("color"))
